const SERVER_URL = "https://myenglishlab.blog/api"; 

document.addEventListener('DOMContentLoaded', () => {
    const loginView = document.getElementById('login-view');
    const mainView = document.getElementById('main-view');
    const status = document.getElementById('status');
    const fullAutoBtn = document.getElementById('fullAutoBtn');
    const loginBtn = document.getElementById('loginBtn');

    // --- RESTORE STATE ---
    chrome.storage.sync.get(['authToken', 'credits', 'fullAutoMode'], (data) => {
        if (data.authToken) {
            showMain(data.credits);
        }
        if (data.fullAutoMode) {
            updateAutoBtn(true);
        }
    });

    // --- LOGIN FLOW ---
    loginBtn.addEventListener('click', async () => {
        const u = document.getElementById('username').value;
        const p = document.getElementById('password').value;
        status.innerText = "Logging in...";
        loginBtn.disabled = true;
        
        try {
            const res = await fetch(`${SERVER_URL}/login.php`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ username: u, password: p })
            });
            const data = await res.json();
            
            if (data.success) {
                chrome.storage.sync.set({ authToken: data.token, credits: data.requests_left });
                showMain(data.requests_left);
            } else {
                status.innerText = "Error: " + (data.message || "Login failed");
            }
        } catch(e) {
            status.innerText = "Connection Error";
        }
        loginBtn.disabled = false;
    });

    // --- LOGOUT HELPER ---
    function handleLogout(reason) {
        chrome.storage.sync.remove(['authToken', 'credits']);
        loginView.style.display = 'block';
        mainView.style.display = 'none';
        status.innerText = reason || "Please login again.";
        status.style.color = "#ff5252";
    }

    // --- FULL AUTO TOGGLE ---
    fullAutoBtn.addEventListener('click', () => {
        chrome.storage.sync.get(['fullAutoMode'], (data) => {
            const newState = !data.fullAutoMode;
            chrome.storage.sync.set({ fullAutoMode: newState });
            updateAutoBtn(newState);
            
            chrome.tabs.query({ active: true, currentWindow: true }, async (tabs) => {
                if(tabs[0]) {
                    chrome.tabs.sendMessage(tabs[0].id, { 
                        action: "update_auto_solve", 
                        enabled: newState 
                    }).catch(() => {}); 
                }
            });
        });
    });

    function updateAutoBtn(isOn) {
        if (isOn) {
            fullAutoBtn.textContent = "Stop Auto-Solver 🛑";
            fullAutoBtn.className = "btn btn-danger";
            status.innerText = "Auto-Solver Active...";
        } else {
            fullAutoBtn.textContent = "Start Auto-Solver 🚀";
            fullAutoBtn.className = "btn btn-primary";
            status.innerText = "Ready";
        }
    }

    // --- ROBUST SOLVE (PING -> INJECT -> EXTRACT) ---
    document.getElementById('solveBtn').addEventListener('click', async () => {
        const storage = await chrome.storage.sync.get(['authToken']);
        const token = storage.authToken;
        
        if (!token) { handleLogout("Please login first."); return; }
        
        status.innerText = "Scanning...";
        status.style.color = "#fff";
        
        const [tab] = await chrome.tabs.query({ active: true, currentWindow: true });
        if (!tab) return;

        try {
            // 1. SCAN: Find frame with content
            const scanResults = await chrome.scripting.executeScript({
                target: { tabId: tab.id, allFrames: true },
                func: () => {
                    const hasQuestions = document.querySelector('.choice_interaction, .gap_match_drag, .input-text, .wrapper-dropdown, .ic-text-gap, .contentblock');
                    const hasTranscript = document.querySelector('[class*="transcript"], [aria-label*="transcript"]');
                    if (hasQuestions || hasTranscript) return { found: true };
                    return null;
                }
            });

            const activeFrame = scanResults.find(r => r.result && r.result.found);

            if (!activeFrame) {
                status.innerText = "No activity found.";
                return;
            }

            const targetFrameId = activeFrame.frameId;
            console.log("Target Frame:", targetFrameId);

            // 2. EXTRACT (With Retry & Injection)
            status.innerText = "Connecting...";
            
            const performExtraction = async (retrying = false) => {
                chrome.tabs.sendMessage(tab.id, { action: "extract_questions" }, { frameId: targetFrameId }, async (response) => {
                    
                    // HANDLE DISCONNECTED CONTENT SCRIPT
                    if (chrome.runtime.lastError) {
                        const err = chrome.runtime.lastError.message;
                        console.warn("Msg Error:", err);

                        if (!retrying && (err.includes("Receiving end does not exist") || err.includes("connection"))) {
                            status.innerText = "Injecting...";
                            try {
                                await chrome.scripting.executeScript({
                                    target: { tabId: tab.id, frameIds: [targetFrameId] },
                                    files: ['content.js']
                                });
                                setTimeout(() => performExtraction(true), 800); // Retry after injection
                            } catch (injErr) {
                                status.innerText = "Injection Failed.";
                                console.error(injErr);
                            }
                        } else {
                            status.innerText = "Connection Failed. Refresh.";
                        }
                        return;
                    }

                    if (response && response.success) {
                        status.innerText = "Thinking...";
                        callSolveAPI(response.data, token, tab.id, targetFrameId);
                    } else {
                        status.innerText = response?.error || "No inputs extracted.";
                    }
                });
            };

            performExtraction();

        } catch (e) {
            console.error(e);
            status.innerText = "Scan Error: " + e.message;
        }
    });

    function callSolveAPI(payload, token, tabId, frameId) {
        chrome.runtime.sendMessage({
            action: "solve_securely",
            payload: payload,
            token: token,
            serverUrl: SERVER_URL,
            model: "gemini-2.5-flash"
        }, (response) => {
            // CHECK FOR AUTH ERROR (401)
            if (response && response.error && response.error.includes("401")) {
                handleLogout("Session expired. Login again.");
                return;
            }

            if (response && response.success) {
                status.innerText = "Applying Answers...";
                chrome.storage.sync.set({ credits: response.creditsLeft });
                document.getElementById('credits').innerText = response.creditsLeft;
                
                chrome.tabs.sendMessage(tabId, { 
                    action: "apply_answers", 
                    answers: response.answers 
                }, { frameId: frameId }, (res) => {
                     status.innerText = `Done! Filled ${res ? res.count : 0} answers.`;
                });
            } else {
                status.innerText = response?.error || "AI Error";
            }
        });
    }

    function showMain(credits) {
        loginView.style.display = 'none';
        mainView.style.display = 'block';
        document.getElementById('credits').innerText = credits;
        status.innerText = "Ready";
        status.style.color = "#fff";
    }
});